#include "gpib.h"
#include "formatio.h"
#include "tt1906.h"

/*= TTi 1906 Multimeter ===================================================*/
/* LabWindows Instrument Driver                                            */
/* Original Release: Nov 1994                                                      */
/* By: John Tothill                                                                    */
/* Originally written in C                                                 */
/* Modification History: None                                              */
/*=========================================================================*/

/*= INSTRUMENT TABLE ======================================================*/
/* address array: contains the GPIB addresses of opened instruments.       */
/* bd array: contains the device descriptors returned by OpenDev.          */
/* instr_cnt: contains the number of instruments open of this model type.  */
/* tt1906_err: the error variable for the instrument module                */
/*=========================================================================*/
static int address[tt1906_MAX_INSTR + 1];
static int bd[tt1906_MAX_INSTR + 1];
static int instr_cnt;
static int tt1906_err;

/*= INTERNAL DATA =========================================================*/
/* cmd is a buffer for GPIB I/O strings */
static char cmd[50];

/*= INSTRUMENT-DEPENDENT COMMAND ARRAYS ===================================*/
static char *resolution[2];
static char *hi_z[2];
static char *functions[7];
static char *ranges[8];
static char *dev[2];
static char *db[2];
static char *axb[2];
static char *limit[2];
static char *mm[2];
static char *null[2];
static char *set_status[3];
static char *set_store[2];
static char *get_status[6];
/*=========================================================================*/

/*= UTILITY ROUTINES ======================================================*/
int tt1906_open_instr (int);
int tt1906_close_instr (int);
int tt1906_invalid_integer_range (int, int, int, int);
int tt1906_invalid_real_range (double, double, double, int);
int tt1906_device_closed (int);
int tt1906_read_data (int, char *, int);
int tt1906_write_data (int, char *, int);
int tt1906_read_data_file (int, char *);
int tt1906_write_data_file (int, char *);
void tt1906_setup_arrays (void);
int tt1906_check_error(int);
/*=========================================================================*/
/* Function: Initialize                                                    */
/* Purpose:  This function opens the instrument, queries the instrument    */
/*           for its ID, and initializes the instrument to a known state.  */
/*=========================================================================*/
int tt1906_init (addr, id_query, rest, instrID)
int addr;
int id_query;    /* If the instrument does not support an ID query and/or */
int rest;        /* a Reset then the id_query and/or the reset controls   */
int *instrID;    /* and variables should be deleted from the function     */
{                /* panel and source code                                 */
    int  ID;

    if (tt1906_invalid_integer_range (addr, 0, 30, -1) != 0)
        return tt1906_err;
    if (tt1906_invalid_integer_range (id_query, 0, 1, -2) != 0)
        return tt1906_err;
    if (tt1906_invalid_integer_range (rest, 0, 1, -3) != 0)
        return tt1906_err;

    ID = tt1906_open_instr (addr);
    if (ID <= 0)
        return tt1906_err;

    if (id_query) {
        if (tt1906_write_data (ID, "*IDN?\012", 6) != 0) {
            tt1906_close_instr (ID);
            return tt1906_err;
        }
        if (tt1906_read_data (ID, cmd, 50) != 0) {
            tt1906_close_instr (ID);
            return tt1906_err;
        }

        Scan (cmd, "THURLBY-THANDAR,1906");
        if (NumFmtdBytes () != 20)  {
            tt1906_err = 223;
            tt1906_close_instr (ID);
            return tt1906_err;
        }

    }

    if (rest) {
        if (tt1906_write_data (ID, "*RST\012", 5) != 0) {
            tt1906_close_instr (ID);
            return tt1906_err;
        }
    }
    tt1906_setup_arrays ();
    *instrID = ID;
    return tt1906_err;
}
/*=========================================================================*/
/* Function: tt1906 config                                                 */
/* Purpose:  Configures the instrument function, range and other minor     */
/*           parameters                                                    */
/*=========================================================================*/
int tt1906_config(instrID, function, range, res, hiz, filter, trigset, cancel)
int instrID;
int function;
int range;
int res;
int hiz;
int filter;
int trigset;
int cancel;
{
int eer;
    if (tt1906_device_closed (instrID))
        return tt1906_err;

    if (tt1906_check_error(instrID) == -1) /* clear the EER */
        return tt1906_err;

    if (tt1906_invalid_integer_range (instrID, 1, tt1906_MAX_INSTR, -1) != 0)
        return tt1906_err;
    if (tt1906_invalid_integer_range (function, 0, 6, -2) != 0)
        return tt1906_err;
    if (tt1906_invalid_integer_range (range, 0, 7, -3) != 0)
        return tt1906_err;
    if (tt1906_invalid_integer_range (res, 0, 1, -4) != 0)
        return tt1906_err;
    if (tt1906_invalid_integer_range (hiz, 0, 1, -5) != 0)
        return tt1906_err;
    if (tt1906_invalid_integer_range (filter, 0, 9, -6) != 0)
        return tt1906_err;
    if (tt1906_invalid_integer_range (trigset, 0, 1, -7) != 0)
        return tt1906_err;
    if (tt1906_invalid_integer_range (cancel, 0, 1, -8) != 0)
        return tt1906_err;

    if (cancel)
        {
        if (tt1906_write_data (instrID, "CANCEL\012", 7) != 0);
           return tt1906_err;
        }

    Fmt(cmd,"%s;%s;TRGSET %d;FILTER %d\012"
                                ,resolution[res],hi_z[hiz],trigset,filter);
    if (tt1906_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return tt1906_err;

    Fmt(cmd,"%s;%s\012",functions[function],ranges[range]);
    if (tt1906_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return tt1906_err;

    eer = tt1906_check_error(instrID);
    if (eer != 200)
        tt1906_err = eer;
    return tt1906_err;
}
/*=========================================================================*/
/* Function: tt1906_measure                                                */
/* Purpose:  reads and returns readings from the instrument                */
/*=========================================================================*/
int tt1906_measure (instrID, read, result_real, result_string)
int instrID;
int read;
double *result_real;
char *result_string;
{
int eer;
    if (tt1906_device_closed (instrID))
        return tt1906_err;

    if (tt1906_invalid_integer_range (instrID, 1, tt1906_MAX_INSTR, -1) != 0)
        return tt1906_err;
    if (tt1906_invalid_integer_range (read, 0, 3, -2) != 0)
        return tt1906_err;

    *result_real = 0;
    result_string[0] = 0;
    switch (read)
    {
    case 0: /* immediate */
        Fmt(cmd,"READ?\012");
        if (tt1906_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
            return tt1906_err;
        if (tt1906_read_data (instrID, result_string, 50) != 0)
            return tt1906_err;
        if (Scan (result_string,"%f",result_real) != 1)
            {
            tt1906_err = 236;
            return tt1906_err;
            }
        break;
    case 1:   /* triggered */
        Fmt(cmd,"TREAD?\012");
        if (tt1906_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
            return tt1906_err;
        break;
    case 2:   /* limits */
        Fmt(cmd,"COMP?\012");
        if (tt1906_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
            return tt1906_err;
        if (tt1906_read_data (instrID, result_string, 50) != 0)
            return tt1906_err;
        break;
    case 3:   /* Min/Max */
        Fmt(cmd,"MM?\012");
        if (tt1906_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
            return tt1906_err;
        if (tt1906_read_data (instrID, result_string, 50) != 0)
            return tt1906_err;
        break;
    }
    return tt1906_err;
}
/*=========================================================================*/
/* Function: tt1906_trigger                                                */
/* Purpose:  Triggers the instrument to take a reading - The instrument    */
/*           must have previously received a triggered read command from   */
/*           the Measure panel or an error will be generated               */
/*=========================================================================*/
int tt1906_trig (instrID, result_real, result_string)
int instrID;
double *result_real;
char *result_string;
{
int eer;
    if (tt1906_device_closed (instrID))
        return tt1906_err;

    if (tt1906_invalid_integer_range (instrID, 1, tt1906_MAX_INSTR, -1) != 0)
        return tt1906_err;

    *result_real = 0;
    result_string[0] = 0;
    Fmt(cmd,"*TRG\012");
    if (tt1906_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return tt1906_err;
    if (tt1906_read_data (instrID, result_string, 50) != 0)
        return tt1906_err;
    if (Scan (result_string,"%f",result_real) != 1)
        tt1906_err = 236;

    return tt1906_err;
}
/*=========================================================================*/
/* Function: tt1906_delta                                                  */
/* Purpose:  Enables or disables the delta% computational function         */
/*=========================================================================*/
int tt1906_delta (instrID, enable, ref_val)
int instrID;
int enable;
double ref_val;
{
int eer;
    if (tt1906_device_closed (instrID))
        return tt1906_err;

    if (tt1906_check_error(instrID) == -1)
        return tt1906_err;

    if (tt1906_invalid_integer_range (instrID, 1, tt1906_MAX_INSTR, -1) != 0)
        return tt1906_err;
    if (tt1906_invalid_integer_range (enable, 0, 2, -2) != 0)
        return tt1906_err;

    if (enable == 2) /* on and config */
        {
        Fmt(cmd,"DEV %f\012",ref_val);
        if (tt1906_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
            return tt1906_err;
        }
    else
        {
        Fmt(cmd,"%s\012",dev[enable]);
        if (tt1906_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
            return tt1906_err;
        }

    eer = tt1906_check_error(instrID);
    if (eer != 200)
        tt1906_err = eer;
    return tt1906_err;
}
/*=========================================================================*/
/* Function: tt1906_db                                                     */
/* Purpose:  Enables or disables the dB computational function             */
/*=========================================================================*/
int tt1906_db (instrID, enable, ref_val)
int instrID;
int enable;
double ref_val;
{
int eer;
    if (tt1906_device_closed (instrID))
        return tt1906_err;

    if (tt1906_check_error(instrID) == -1)
        return tt1906_err;

    if (tt1906_invalid_integer_range (instrID, 1, tt1906_MAX_INSTR, -1) != 0)
        return tt1906_err;
    if (tt1906_invalid_integer_range (enable, 0, 2, -2) != 0)
        return tt1906_err;

    if (enable == 2) /* on and config */
        {
        Fmt(cmd,"DB %f\012",ref_val);
        if (tt1906_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
            return tt1906_err;
        }
    else
        {
        Fmt(cmd,"%s\012",db[enable]);
        if (tt1906_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
            return tt1906_err;
        }

    eer = tt1906_check_error(instrID);
    if (eer != 200)
        tt1906_err = eer;
    return tt1906_err;
}
/*=========================================================================*/
/* Function: tt1906_axb                                                     */
/* Purpose:  Enables or disables Ax+B computational function             */
/*=========================================================================*/
int tt1906_axb (instrID, enable, a , b)
int instrID;
int enable;
double a;
double b;
{
int eer;
    if (tt1906_device_closed (instrID))
        return tt1906_err;

    if (tt1906_check_error(instrID) == -1)
        return tt1906_err;

    if (tt1906_invalid_integer_range (instrID, 1, tt1906_MAX_INSTR, -1) != 0)
        return tt1906_err;
    if (tt1906_invalid_integer_range (enable, 0, 2, -2) != 0)
        return tt1906_err;
    if (tt1906_invalid_real_range (a, -999999, +999999, -3) != 0)
        return tt1906_err;

    if (enable == 2) /* on and config */
        {
        Fmt(cmd,"AXB %f,%f\012",a,b);
        if (tt1906_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
            return tt1906_err;
        }
    else
        {
        Fmt(cmd,"%s\012",axb[enable]);
        if (tt1906_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
            return tt1906_err;
        }

    eer = tt1906_check_error(instrID);
    if (eer != 200)
        tt1906_err = eer;
    return tt1906_err;
}
/*=========================================================================*/
/* Function: tt1906_limits                                                 */
/* Purpose:  Enables or disables Limits computational function             */
/*=========================================================================*/
int tt1906_limits (instrID, enable, low , hi)
int instrID;
int enable;
double low;
double hi;
{
int eer;
    if (tt1906_device_closed (instrID))
        return tt1906_err;

    if (tt1906_check_error(instrID) == -1)
        return tt1906_err;

    if (tt1906_invalid_integer_range (instrID, 1, tt1906_MAX_INSTR, -1) != 0)
        return tt1906_err;
    if (tt1906_invalid_integer_range (enable, 0, 2, -2) != 0)
        return tt1906_err;

    if (enable == 2) /* on and config */
        {
        Fmt(cmd,"LIMITS %f,%f\012",low,hi);
        if (tt1906_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
            return tt1906_err;
        }
    else
        {
        Fmt(cmd,"%s\012",limit[enable]);
        if (tt1906_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
            return tt1906_err;
        }

    eer = tt1906_check_error(instrID);
    if (eer != 200)
        tt1906_err = eer;
    return tt1906_err;
}
/*=========================================================================*/
/* Function: tt1906_minmax                                                 */
/* Purpose:  Enables or disables Min/Max computational function            */
/*=========================================================================*/
int tt1906_minmax (instrID, enable)
int instrID;
int enable;
{
int eer;
    if (tt1906_device_closed (instrID))
        return tt1906_err;

    if (tt1906_check_error(instrID) == -1)
        return tt1906_err;

    if (tt1906_invalid_integer_range (instrID, 1, tt1906_MAX_INSTR, -1) != 0)
        return tt1906_err;
    if (tt1906_invalid_integer_range (enable, 0, 1, -2) != 0)
        return tt1906_err;

    Fmt(cmd,"%s\012",mm[enable]);
    if (tt1906_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return tt1906_err;

    return tt1906_err;
}
/*=========================================================================*/
/* Function: tt1906_null                                                   */
/* Purpose:  Enables or disables Null computational function            */
/*=========================================================================*/
int tt1906_null (instrID, enable)
int instrID;
int enable;
{
int eer;
    if (tt1906_device_closed (instrID))
        return tt1906_err;

    if (tt1906_invalid_integer_range (instrID, 1, tt1906_MAX_INSTR, -1) != 0)
        return tt1906_err;
    if (tt1906_invalid_integer_range (enable, 0, 1, -2) != 0)
        return tt1906_err;

    Fmt(cmd,"%s\012",null[enable]);
    if (tt1906_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return tt1906_err;

    eer = tt1906_check_error(instrID);
    if (eer != 200)
        tt1906_err = eer;
    return tt1906_err;
}
/*=========================================================================*/
/* Function: tt1906_set_stat                                             */
/* Purpose:  Set the value in the Service Request Enable, Event Status     */
/*           Enable or Parallel Poll Enable register                       */
/*=========================================================================*/
int tt1906_set_stat (instrID, reg, val)
int instrID;
int reg;
int val;
{
    if (tt1906_invalid_integer_range (instrID, 1, tt1906_MAX_INSTR, -1) != 0)
        return tt1906_err;
    if (tt1906_invalid_integer_range (reg, 0, 2, -2) != 0)
        return tt1906_err;
    if (tt1906_invalid_integer_range (val, 0, 255, -3) != 0)
        return tt1906_err;

    if (tt1906_device_closed (instrID))
        return tt1906_err;

    Fmt (cmd, "%s %d\012", set_status[reg],val);
    if (tt1906_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return tt1906_err;

    return tt1906_err;
}
/*=========================================================================*/
/* Function: tt1906_store                                             */
/* Purpose:  Allows saving to or recalling from the instruments local non- */
/*           volatile stores                                               */
/*=========================================================================*/
int tt1906_store (instrID, op, storeno)
int instrID;
int op;
int storeno;
{
    if (tt1906_invalid_integer_range (instrID, 1, tt1906_MAX_INSTR, -1) != 0)
        return tt1906_err;
    if (tt1906_invalid_integer_range (op, 0, 1, -1) != 0)
        return tt1906_err;
    if (tt1906_invalid_integer_range (storeno, 0, 5, -2) != 0)
        return tt1906_err;

    if (tt1906_device_closed (instrID))
        return tt1906_err;

    Fmt (cmd, "%s %d\012", set_store[op],storeno);
    if (tt1906_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return tt1906_err;

    return tt1906_err;
}
/*=========================================================================*/
/* Function: tt1906_get stat                                               */
/* Purpose:  Reads the value in the selected iEEE 488.2 status register    */
/*=========================================================================*/
int tt1906_get_stat (instrID, reg, val)
int instrID;
int reg;
int *val;
{
    if (tt1906_invalid_integer_range (instrID, 1, tt1906_MAX_INSTR, -1) != 0)
        return tt1906_err;
    if (tt1906_invalid_integer_range (reg, 0, 5, -2) != 0)
        return tt1906_err;

    if (tt1906_device_closed (instrID))
        return tt1906_err;

    Fmt (cmd, "%s\012", get_status[reg]);
    if (tt1906_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return tt1906_err;
    if (tt1906_read_data (instrID, cmd, 50) != 0)
        return tt1906_err;
    if (Scan (cmd, "%d",val) != 1)
        tt1906_err = 236;

    return tt1906_err;
}
/*=========================================================================*/
/* Function: tt1906_send_cmd                                                */
/* Purpose:  Sends the data in the supplied buffer to the instrument       */
/*           Use for commands not available in this driver - e.g. inc and  */
/*           dec commands.												   */
/*=========================================================================*/
int tt1906_send_cmd (instrID, command)
int instrID;
char *command;
{
char output_buf[200];
    if (tt1906_invalid_integer_range (instrID, 1, tt1906_MAX_INSTR, -1) != 0)
        return tt1906_err;

    if (tt1906_device_closed (instrID))
        return tt1906_err;

    Fmt(output_buf,"%s",command);
    if (tt1906_write_data(instrID,command,NumFmtdBytes()) != 0)
        return tt1906_err;

    return tt1906_err;
}
/*=========================================================================*/
/* Function: tt1906_get_resp                                               */
/* Purpose:  Gets a response to a previous query. Used in conjuction with  */
/*           the send command function                                     */
/*=========================================================================*/
int tt1906_get_resp (instrID, response, nbytes)
int instrID;
char *response;
int *nbytes;
{
    if (tt1906_invalid_integer_range (instrID, 1, tt1906_MAX_INSTR, -1) != 0)
        return tt1906_err;

    if (tt1906_device_closed (instrID))
        return tt1906_err;

    if (tt1906_read_data(instrID,response,2000))
        return tt1906_err;
    *nbytes = ibcnt;

    return tt1906_err;
}
/*=========================================================================*/
/* Function: tt1906_file                                                   */
/* Purpose:  Send or receive files                                         */
/*=========================================================================*/
int tt1906_file (instrID, op, fname)
int instrID;
int op;
char *fname;
{
    if (tt1906_invalid_integer_range (instrID, 1, tt1906_MAX_INSTR, -1) != 0)
        return tt1906_err;
    if (tt1906_invalid_integer_range (op, 0, 1, -2) != 0)
        return tt1906_err;

    if (tt1906_device_closed (instrID))
        return tt1906_err;

    if (op == 1) /* get to file from tt1906 */
        {
        if (tt1906_read_data_file(instrID,fname))
            return tt1906_err;
        }
    else /* send from file to tt1906 */
        if (tt1906_write_data_file(instrID,fname))
            return tt1906_err;

    return tt1906_err;
}
/*=========================================================================*/
/* Function: Close                                                         */
/* Purpose:  This function closes the instrument.                          */
/*=========================================================================*/
int tt1906_close (instrID)
int instrID;
{
    if (tt1906_invalid_integer_range (instrID, 1, tt1906_MAX_INSTR, -1) != 0)
        return tt1906_err;

    if (tt1906_device_closed (instrID))
        return tt1906_err;

    tt1906_close_instr (instrID);
    return tt1906_err;
}
/*= UTILITY ROUTINES ======================================================*/

/*=========================================================================*/
/* Function: Open Instrument                                               */
/* Purpose:  This function locates and initializes an entry in the         */
/*           Instrument Table and the GPIB device table for the            */
/*           instrument.  If successful, the instrument ID is returned,    */
/*           else a -1 is returned.  The size of the Instrument Table can  */
/*           be changed in the include file by altering the constant       */
/*           tt1906_MAX_INSTR.                                             */
/*=========================================================================*/
int tt1906_open_instr (addr)
int addr;
{
    int i, instrID;

    instrID = 0;

/* Check to see if the instrument is already in the Instrument Table. */

    for (i=1; i<= tt1906_MAX_INSTR; i++)
        if (address[i] == addr) {
            instrID = i;
            i = tt1906_MAX_INSTR;
         }

/* If it is not in the instrument table, open an entry for the instrument. */

    if (instrID <= 0)
        for (i=1; i<= tt1906_MAX_INSTR; i++)
            if (address[i] == 0) {
                instrID = i;
                i = tt1906_MAX_INSTR;
             }

/* If an entry could not be opened in the Instrument Table, return an error.*/

    if (instrID <= 0) {
        tt1906_err = 220;
        return -1;
     }

/*  If the device has not been opened in the GPIB device table (bd[ID] = 0),*/
/*  then open it.                                                           */

    if (bd[instrID] <= 0)    {
        if (instr_cnt <= 0)
            CloseInstrDevs("tt1906");
        bd[instrID] = OpenDev ("", "tt1906");
        if (bd[instrID] <= 0)    {
            tt1906_err = 220;
            return -1;
        }
        instr_cnt += 1;
        address[instrID] = addr;
     }

/*  Change the primary address of the device    */

    if (ibpad (bd[instrID], addr) < 0)   {
        tt1906_err = 233;
        return -1;
    }

    return instrID;
}
/*=========================================================================*/
/* Function: Close Instrument                                              */
/* Purpose:  This function closes the instrument by removing it from the   */
/*           GPIB device table and setting the address and the bd to zero  */
/*           in the Instrument Table.  The return value is equal to the    */
/*           global error variable.                                        */
/*=========================================================================*/
int tt1906_close_instr (instrID)
int instrID;
{
    if (bd[instrID] != 0) {
        CloseDev (bd[instrID]);
        bd[instrID] = 0;
        address[instrID] = 0;
        instr_cnt -= 1;
     }
    else
        tt1906_err = 221;

    return tt1906_err;
 }
/*=========================================================================*/
/* Function: Invalid Integer Range                                         */
/* Purpose:  This function checks an integer to see if it lies between a   */
/*           minimum and maximum value.  If the value is out of range, set */
/*           the global error variable to the value err_code.  If the      */
/*           value is OK, error = 0.                                       */
/*=========================================================================*/
int tt1906_invalid_integer_range (val, min, max, err_code)
int val;
int min;
int max;
int err_code;
{
  if ((val < min) || (val > max)) {
    tt1906_err = err_code;
    return -1;
  }
  return 0;
}
/*=========================================================================*/
/* Function: Invalid Real Range                                            */
/* Purpose:  This function checks a real number to see if it lies between  */
/*           a minimum and maximum value.  If the value is out of range,   */
/*           set the global error variable to the value err_code.  If the  */
/*           value is OK, error = 0.                                       */
/*=========================================================================*/
int tt1906_invalid_real_range (val, min, max, err_code)
double val;
double min;
double max;
int err_code;
{
  if ((val < min) || (val > max)) {
    tt1906_err = err_code;
    return -1;
  }
  return 0;
}
/*=========================================================================*/
/* Function: Device Closed                                                 */
/* Purpose:  This function checks to see if the module has been            */
/*           initialized.  If the device has not been opened, a 1 is       */
/*           returned, 0 otherwise.                                        */
/*=========================================================================*/
int tt1906_device_closed (instrID)
int instrID;
{
    if (bd[instrID] <= 0)  {
        tt1906_err = 232;
        return -1;
    }
    return 0;
}
/*=========================================================================*/
/* Function: Read Data                                                     */
/* Purpose:  This function reads a buffer of data from the instrument. The */
/*           return value is equal to the global error variable.           */
/*=========================================================================*/
int tt1906_read_data (instrID, buf, cnt)
int instrID;
char *buf;
int cnt;
{
    if (ibrd(bd[instrID], buf, (long)cnt) <= 0)
        tt1906_err = 231;
    else
        tt1906_err = 0;

    return tt1906_err;
}
/*=========================================================================*/
/* Function: Write Data                                                    */
/* Purpose:  This function writes a buffer of data to the instrument. The  */
/*           return value is equal to the global error variable.           */
/*=========================================================================*/
int tt1906_write_data (instrID, buf, cnt)
int instrID;
char *buf;
int cnt;
{
    if (ibwrt(bd[instrID], buf, (long)cnt) <= 0)
        tt1906_err = 230;
    else
        tt1906_err = 0;

    return tt1906_err;
}
/*=========================================================================*/
/* Function: Read Data File                                                */
/* Purpose:  This function reads a buffer of data from the instrument and  */
/*           stores it to the file specified by "filename".  Filename must */
/*           either be a string, such as "C:\lw\instr\file" or a pointer   */
/*           to such a string.  The return value is equal to the global    */
/*           error variable.                                               */
/*=========================================================================*/
int tt1906_read_data_file (instrID, filename)
int instrID;
char *filename;
{
    if (ibrdf (bd[instrID], filename) <= 0)
        tt1906_err = 229;
    else
        tt1906_err = 0;

    return tt1906_err;
}
/*=========================================================================*/
/* Function: Write Data File                                               */
/* Purpose:  This function writes a buffer of data from the file specified */
/*           by "filename" to the instrument. Filename must either be a    */
/*           string, such as "C:\lw\instr\file" or a pointer to such a     */
/*           string.  The return value is equal to the global error        */
/*           variable.                                                     */
/*=========================================================================*/
int tt1906_write_data_file (instrID, filename)
int instrID;
char *filename;
{
    if (ibwrtf (bd[instrID], filename) <= 0)
        tt1906_err = 228;
    else
        tt1906_err = 0;

    return tt1906_err;
}
/*=========================================================================*/
/* Function: tt1906_check_error                                            */
/* Purpose:  reads the event status register, looks for exec error, if true*/
/*           reads exec error register and returns the error               */
/*=========================================================================*/
int tt1906_check_error(instrID)
int instrID;
{
int eer;
    tt1906_err = 0;
    Fmt(cmd,"EER?\012");
    if (tt1906_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return -1;
    if (tt1906_read_data (instrID, cmd, 50) != 0)
        return -1;
    if (Scan (cmd, "%d",&eer) != 1)
        {
        tt1906_err = 236;
        return -1;
        }
    eer = eer + 200;
    return eer;
}
/*=========================================================================*/
/* Function: Setup Arrays                                                  */
/* Purpose:  This function is called by the init routine to initialize     */
/*           global arrays.                                                */
/*           This routine should be modified for each instrument to        */
/*           include instrument-dependent commmand arrays.                 */
/*=========================================================================*/
void tt1906_setup_arrays ()
{
    resolution[0] = "SLOW";
    resolution[1] = "FAST";

    hi_z[0] = "LOZ";
    hi_z[1] = "HIZ";

    functions[0] = "VDC";
    functions[1] = "VAC";
    functions[2] = "ADC";
    functions[3] = "AAC";
    functions[4] = "A10DC";
    functions[5] = "A10AC";
    functions[6] = "OHMS";

    ranges[0] = "RANGE 0";
    ranges[1] = "RANGE 1";
    ranges[2] = "RANGE 2";
    ranges[3] = "RANGE 3";
    ranges[4] = "RANGE 4";
    ranges[5] = "RANGE 5";
    ranges[6] = "AUTO";
    ranges[7] = "MAN";

    dev[0] = "DEVOFF";
    dev[1] = "DEV";

    db[0] = "DBOFF";
    db[1] = "DB";

    axb[0] = "AXBOFF";
    axb[1] = "AXB";

    limit[0] = "LIMOFF";
    limit[1] = "LIMITS";

    mm[0] = "MMOFF";
    mm[1] = "MMON";

    null[0] = "NULLOFF";
    null[1] = "NULL";

    set_status[0] = "*SRE";
    set_status[1] = "*ESE";
    set_status[2] = "*PRE";

    set_store[0] = "*RCL";
    set_store[1] = "*SAV";

    get_status[0] = "*STB?";
    get_status[1] = "*ESR?";
    get_status[2] = "*SRE?";
    get_status[3] = "*ESE?";
    get_status[4] = "*PRE?";
    get_status[5] = "EER?";
}
/*= THE END ===============================================================*/
